package mcfall.math;

/**
 * <p>
 * <B>Matrix</B> represents the mathematical notion of a matrix, which is an <i>m by n</i>
 * rectangular grid of data values.  While an ideal matrix implementation would allow
 * storage of any type of data that has both add and scalar multiply operations, our
 * implementation stays simple and stores only double values.
 * </p>
 * <p>
 * The size of the matrix cannot be changed.  Furthermore, any operations on the matrix return <b>new</b> matrix objects.
 * The only way to change the contents of the matrix is through the <i>setValueAt</i> method.
 * </p>
 * <p>
 * The range of index values for both rows and columns can be customized; by default
 * the first index is 1.  The <i>setFirstRowIndex</i> and <i>setFirstColumnIndex</i>
 * allow the first index value to be set to an arbitrary integer value, including
 * negative values and zero.  The methods <i>getLastRowIndex</i> and
 * <i>getLastColumnIndex</i> can be used to retrieve the index of the last
 * row and column, respectively.
 * </p>
 * 
 * @author mcfall
 * @version 1.0
 * @date June 5, 2006
 */
public class Matrix {
	
	/** The number of rows for this matrix  *. */
	private int numberOfRows;
	
	/** The number of columns for this matrix *. */
	private int numberOfColumns;
	
	/** The actual data values  *. */
	double[][] data;
	
	/** The index value used to access the first row in the matrix  *. */
	private int firstRowIndex;
	
	/** The index value used to access the first column in the matrix *. */
	private int firstColumnIndex;
	
	/**
	 * Returns an identity matrix of the requested size.
	 * 
	 * @param size the number of rows and columns for the matrix
	 * 
	 * @return a new Matrix instance whose values are 0 everywhere except along
	 * the diagonal, which have the value 1
	 */
	public static SquareMatrix createIdentityMatrix (int size) {
		double[][] data = new double[size][size];
		for (int i = 0; i < size; i++) {
			data[i][i] = 1.0;
		}
		return new SquareMatrix (size, data);
	}
	
	/**
	 * Creates a 4x4 translation matrix, whose application results in a translation by the amount
	 * (tx, ty, tz).
	 * 
	 * @param tx the amount to translate by in the x direction
	 * @param ty the amount to translate by in the y direction
	 * @param tz the amount to translate by in the z direction
	 * 
	 * @return a new 4x4 translation matrix
	 */
	public static Matrix createTranslationMatrix (double tx, double ty, double tz) {
		Matrix result = createIdentityMatrix(4);
		result.setValueAt(1, 4, tx);
		result.setValueAt(2, 4, ty);
		result.setValueAt(3, 4, tz);
		return result;
	}
	
	/**
	 * Creates a 4x4 scaling matrix, whose application results in a scaling around the origin.
	 * 
	 * @param sx the amount to scale in the x direction
	 * @param sy the amount to scale in the y direction
	 * @param sz the amount to scale in the z direction
	 * 
	 * @return a new 4x4 scaling matrix
	 */
	public static Matrix createScalingMatrix(double sx, double sy, double sz) {
		Matrix result = new Matrix (4, 4);
		result.setValueAt(1, 1, sx);
		result.setValueAt(2, 2, sy);
		result.setValueAt(3, 3, sz);
		result.setValueAt(4, 4, 1);
		return result;		
	}
	
	/**
	 * Creates a 4x4 rotation matrix, whose application results in rotating by the specified angle around the
	 * axis of rotation specified.
	 * 
	 * @param angleDeg the angle, in degrees, to rotate by
	 * @param axisOfRotation the vector around which to rotate
	 * 
	 * @return a new 4x4 rotation matrix
	 */
	public static Matrix createRotationMatrix (double angleDeg, Vector axisOfRotation) {
		double radians = Math.toRadians(angleDeg);
		double c = Math.cos(radians);
		double s = Math.sin(radians);
		Vector axis  = axisOfRotation.normalize();
		double ux = axis.getValueAt(axis.getFirstIndex());
		double uy = axis.getValueAt(axis.getFirstIndex()+1);
		double uz = axis.getValueAt(axis.getFirstIndex()+2);
		
		Matrix result = createIdentityMatrix(4);
		result.setValueAt (1, 1, c+(1-c)*ux*ux);
		result.setValueAt (1, 2, (1-c)*uy*ux-s*uz);
		result.setValueAt (1, 3, (1-c)*uz*ux+s*uy);
		
		result.setValueAt (2, 1, (1-c)*ux*uy+s*uz);
		result.setValueAt (2, 2, c+(1-c)*uy*uy);
		result.setValueAt (2, 3, (1-c)*uz*uy-s*ux);
		
		result.setValueAt (3, 1, (1-c)*ux*uz -s*uy);
		result.setValueAt (3, 2, (1-c)*uy*uz+s*ux);
		result.setValueAt (3, 3, c+(1-c)*uz*uz);
		
		return result;
	}
	
	/**
	 * Creates a 4x4 rotation matrix, whose application results in rotating by the specified angle around the X axis.
	 * 
	 * @param angleDeg the angle, in degrees, to rotate by
	 * 
	 * @return a new 4x4 rotation matrix
	 */
	public static Matrix createRotationAroundXMatrix(double angleDeg) {		
		return Matrix.createRotationMatrix(angleDeg, new ColumnVector (4, new double[] {1.0, 0.0, 0.0, 0.0}));
	}

	/**
	 * Creates a 4x4 rotation matrix, whose application results in rotating by the specified angle around the Y axis.
	 * 
	 * @param angleDeg the angle, in degrees, to rotate by
	 * 
	 * @return a new 4x4 rotation matrix
	 */
	public static Matrix createRotationAroundYMatrix(double angleDeg) {
		return Matrix.createRotationMatrix(angleDeg, new ColumnVector (4, new double[] {0.0, 1.0, 0.0, 0.0}));
	}
	
	/**
	 * Creates a 4x4 rotation matrix, whose application results in rotating by the specified angle around the Z axis.
	 * 
	 * @param angleDeg the angle, in degrees, to rotate by
	 * 
	 * @return a new 4x4 rotation matrix
	 */
	public static Matrix createRotationAroundZMatrix(double angleDeg) {
		return Matrix.createRotationMatrix(angleDeg, new ColumnVector (4, new double[] {0.0, 0.0, 1.0, 0.0}));
	}
	
	/**
	 * Constructs a 4 by 4 matrix with all values initially set to 0.  Rows and columns are 1-based (that is, the
	 * first row/column is numbered 1)
	 */
	public Matrix () {
		this (4, 4);
	}
	
	/**
	 * Constructs a matrix instance that holds the specified number of rows and columns of double values.
	 * All values are initially set to 0; rows and columns are 1-based (that is, the first row/column
	 * is numbered 1)
	 * 
	 * @param rows the number of rows in the matrix
	 * @param columns the number of columns in the matrix
	 */
	public Matrix (int rows, int columns) {
		this (rows, columns, null);
	}
	
	/**
	 * Constructs a matrix that holds the specified number of rows and columns of double values.
	 * The values are initialized to be the values in the array <i>values</i>.  Rows and columns
	 * are 1-based (that is, the first row/column is numbered 1)
	 * 
	 * @param rows the number of rows in the matrix
	 * @param columns the number of columns in the matrix
	 * @param values the initial values for the matrix entries; the first dimension of the array
	 * corresponds to a row of the matrix, while the second dimension holds the columns
	 * 
	 * @throws IllegalArgumentException the illegal argument exception
	 * 
	 * @thorws IllegalArgumentException if the dimensions of the array <i>values</i> do not match
	 * the values of <i>rows</i> and <i>columns</i>
	 */
	public Matrix (int rows, int columns, double[][] values) throws IllegalArgumentException {
		//  Note that Java guarantees initial values to be 0, so we can just create values as below
		//  to get the correct initial behavior
		if (values == null) {
			values = new double[rows][columns];
		}
		
		if (rows != values.length) {
			throw new IllegalArgumentException ("Number of rows in matrix (" + rows + ") differs from number of rows in initial data values array (" + values.length + ")");
		}
		
		if (columns != values[0].length) {
			throw new IllegalArgumentException ("Number of columns in matrix (" + columns + ") differs from number of columns in initial data values array (" + values[0].length + ")");
		}
		
		data = new double[rows][columns];
		firstColumnIndex = 1;
		firstRowIndex = 1;
		numberOfRows = rows;
		numberOfColumns = columns;
		
		for (int row = 0; row < rows; row++) {
			for (int col = 0; col < columns; col++) {
				data[row][col] = values[row][col]; 
			}
		}		
	}
	
	/**
	 * Determine the number of rows in this matrix.
	 * 
	 * @return an integer representing the number of rows in the matrix
	 */
	public int getNumberOfRows () {
		return numberOfRows;
	}
	
	/**
	 * Determine the number of columns in this matrix.
	 * 
	 * @return an integer representing the number of columns in the matrix
	 */
	public int getNumberOfColumns () {
		return numberOfColumns;
	}
	
	/**
	 * Sets the index that will be used to access the first column in the matrix.
	 * 
	 * @param indexValue the new integer value of the first column
	 */
	public void setFirstColumnIndex (int indexValue) {
		firstColumnIndex = indexValue;
	}
	
	/**
	 * Sets the index that will be used to access the first row in the matrix.
	 * 
	 * @param indexValue the new integer value of the first row
	 */
	public void setFirstRowIndex (int indexValue) {
		firstRowIndex = indexValue;
	}
	
	/**
	 * Retrieves the index of the first row.
	 * 
	 * @return an integer representing the index of the first row in the matrix
	 */
	public int getFirstRowIndex () {
		return firstRowIndex;
	}

	/**
	 * Retrieves the index of the last row, based on the setting of the first row index
	 * and the number of rows.
	 * 
	 * @return an integer representing the index of the last row in the matrix
	 */
	public int getLastRowIndex () {
		return getFirstRowIndex() + numberOfRows-1;
	}
	
	/**
	 * Retrieves the index of the first column.
	 * 
	 * @return an integer representing the index of the first column in the matrix
	 */
	public int getFirstColumnIndex () {
		return firstColumnIndex;
	}
	
	/**
	 * Retrieves the index of the last column, based on the setting of the first column index
	 * and the number of columns.
	 * 
	 * @return an integer representing the index of the last column in the matrix
	 */
	public int getLastColumnIndex () {
		return getFirstColumnIndex() + numberOfColumns-1;
	}
	
	/**
	 * Retrieves the value at row,column in the matrix.
	 * 
	 * @param row the row of the desired data value; the index of the first row is
	 * determined by getFirstRowIndex
	 * @param column the column of the desired data value; the index of the first column
	 * is determined by getFirstColumnIndex
	 * 
	 * @return the double value at location row,column
	 * 
	 * @throws IndexOutOfBoundsException if any of row &lt; getFirstRowIndex(), row &gt; getLastRowIndex(),
	 * column &lt; getFirstColumnIndex(), or column &gt; getLastColumnIndex() are true
	 */
	public double getValueAt (int row, int column) throws IndexOutOfBoundsException {
		if (!isValidRow (row)) {
			throwInvalidRowException(row);
		}
		
		if (!isValidColumn (column)) {
			throwInvalidColumnException(column);
		}
				
		return data[getActualRow(row)][getActualColumn(column)];		
	}
	
	/**
	 * Sets the value at row,column in the matrix.
	 * 
	 * @param row the row of the desired data value to be changed; the index of the first row is
	 * determined by getFirstRowIndex
	 * @param column the column of the desired data value to be changed; the index of the first column
	 * is determined by getFirstColumnIndex
	 * @param value the value
	 * 
	 * @return the <b>old</b> double value at location row,column
	 * 
	 * @throws IndexOutOfBoundsException if any of row &lt; getFirstRowIndex(), row &gt; getLastRowIndex(),
	 * column &lt; getFirstColumnIndex(), or column &gt; getLastColumnIndex() are true
	 */
	public double setValueAt (int row, int column, double value) throws IndexOutOfBoundsException {
		if (!isValidRow(row)) {
			throwInvalidRowException (row);			
		}
		
		if (!isValidColumn(column)) {
			throwInvalidColumnException(column);
		}
		
		double lastValue = getValueAt (row, column);
		data[getActualRow(row)][getActualColumn(column)] = value;
		return lastValue;
	}
	
	

	/**
	 * Creates a new matrix that is the transpose of the current matrix
	 * If the returned matrix is named <i>result</i>, the following conditions should hold
	 * <ul>
	 * <li>this.getFirstColumnIndex() = result.getFirstRowIndex ()</li>
	 * <li>this.getLastColumnIndex() = result.getLastRowIndex ()</li>
	 * <li>this.getFirstRowIndex() = result.getFirstColumnIndex ()</li>
	 * <li>this.getLastRowIndex() = result.getLastColumnIndex ()</li>
	 * <li>this.getValueAt(row,col) = result.getValueAt(col,row)</li>
	 * </ul>
	 * 
	 * @return a new columns x rows matrix whose rows are the columns of
	 * this matrix, and whose columns are the rows of this matrix
	 */
	public Matrix transpose () {
		Matrix newMatrix = new Matrix (getNumberOfColumns(), getNumberOfRows());
		newMatrix.setFirstColumnIndex(getFirstRowIndex());
		newMatrix.setFirstRowIndex(getFirstColumnIndex());
		
		for (int r = getFirstRowIndex(); r <= getLastRowIndex(); r++) {
			for (int c = getFirstColumnIndex(); c <= getLastColumnIndex(); c++) {
				newMatrix.setValueAt(c, r, getValueAt (r, c));
			}
		}
		
		return newMatrix;
	}
	
	/**
	 * Computes the product of this matrix and <i>factor</i> by forming the product <i>factor</i> x <i>this</i>.
	 * 
	 * @param factor a matrix to multiply by
	 * 
	 * @return a new matrix which is the result of evaluating <i>factor</i> x <i>this</i>.  The first row and column
	 * indices will be the same as those of <b>this matrix</b>.
	 * 
	 * @throws IncompatibleMatrixException if the matrix <i>factor</i> is not compatible with this matrix
	 */
	public Matrix premultiply (Matrix factor) {
		return factor.postmultiply(this);
	}
		
	/**
	 * Computes the product of this matrix and <i>factor</i> by forming the product <i>this</i> x <i>factor</i>.
	 * 
	 * @param factor a matrix to multiply by
	 * 
	 * @return a new matrix which is the result of evaluating <i>this</i> x <i>factor</i>.  The first row and column
	 * indices will be the same as those of <b>this matrix</b>.
	 * 
	 * @throws IncompatibleMatrixException if the matrix <i>factor</i> is not compatible with this matrix
	 */
	public Matrix postmultiply (Matrix factor) {
		
		/*
		 * Check to ensure these matrices are compatible
		 */
		if (this.getNumberOfColumns() != factor.getNumberOfRows()) {
			throw new IncompatibleMatrixException (this, factor);
		}
		
		int resultNumberOfRows = this.getNumberOfRows();
		int resultNumberOfColumns = factor.getNumberOfColumns();
		Matrix result = new Matrix (resultNumberOfRows, resultNumberOfColumns);
		
		//  Set these temporarily to 0, because it makes the implementation easier
		result.setFirstColumnIndex(0);
		result.setFirstRowIndex(0);
		
		double[][] otherData = factor.data;
		for (int r=0; r < resultNumberOfRows; r++) {
			for (int c = 0; c < resultNumberOfColumns; c++) {
				double sum = 0;
				for (int column=0; column < getNumberOfColumns(); column++) {
					double thisValue = data[r][column];
					double factorValue = otherData[column][c];
					//double factorValue = factor.data[column][c];
					sum += thisValue * factorValue;
				}
				result.setValueAt(r, c, sum);
			}
		}
		
		//  Set these values to what is expected before exiting this method
		result.setFirstColumnIndex(getFirstColumnIndex());
		result.setFirstRowIndex(getFirstRowIndex());
		
		
		return result;
	}
	
	/**
	 * Multiples each entry in this matrix by a scalar value.
	 * 
	 * @param scalar the value to multiply by
	 * 
	 * @return a new Matrix <b>R</b> whose values satisfy R<sub>r,c</sub> = scalar * this<sub>r,c</sub>.
	 * The first row and column indices will be the same as those of this matrix.
	 */
	public Matrix scalarMultiply (double scalar) {
		Matrix result = new Matrix (getNumberOfRows(), getNumberOfColumns());
		result.setFirstColumnIndex(getFirstColumnIndex());
		result.setFirstRowIndex(getFirstRowIndex());
		for (int row = getFirstRowIndex(); row <= getLastRowIndex(); row++) {
			for (int col = getFirstColumnIndex(); col <= getLastColumnIndex(); col++) {
				result.setValueAt(row, col, scalar*getValueAt(row, col));
			}
		}
		return result;
	}
	
	/**
	 * Returns the values stored in this array as an array of doubles.
	 * 
	 * @return a double array containing a copy of the data values for this array
	 */
	public double[][] toArray () {
		double[][] results = new double[getNumberOfRows()][getNumberOfColumns()];
		for (int r = 0; r < getNumberOfRows(); r++) {
			for (int c = 0; c < getNumberOfColumns(); c++) {
				results[r][c] = data[r][c];
			}
		}
		return results;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString () {
		StringBuffer buffer = new StringBuffer ();
		for (int r=0; r < getNumberOfRows(); r++) {
			for (int c = 0; c < getNumberOfColumns(); c++) {
				buffer.append (data[r][c]);
				buffer.append (" ");
			}
			buffer.append ("\n");
		}
		return buffer.toString();
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	/**
	 * Determines whether this object should be considered equal to the 
	 * paramemter
	 * @param o the object to compare with
	 * @return true if <i>o</i> is a matrix with the same size as this matrix,
	 * and the individual elements are equal; false otherwise
	 */
	public boolean equals (Matrix o) {
		Matrix other = (Matrix) o;
		if (other.getNumberOfColumns() != getNumberOfColumns ()) {
			return false;
		}
		
		if (other.getNumberOfRows() != getNumberOfRows()) {
			return false;
		}
		
		//  Here we know they are the same size, so we start comparing
		//  Rather than using logical indices, we use physical ones to make
		//  things simpler
		for (int row = 0; row < data.length; row++) {
			for (int column = 0; column < data[0].length; column++) {
				if (data[row][column] != other.data[row][column]) {
					return false;
				}
			}
		}
		
		return true;
	}
			
	/**
	 * Computes the inverse of this matrix; the inverse of a matrix is the
	 * unique matrix M such that this x M = identity.
	 * 
	 * @return a new Matrix object that is the inverse of this matrix
	 * 
	 * @throws NotInvertibleException if this matrix is not invertible
	 */
	public Matrix invert() throws NotInvertibleException{
		
		if (!isInvertible()) {
			throw new NotInvertibleException (this);
		}
		
		/*  
		 * First, create a new matrix which contains the same number of
		 * rows as this matrix, and has the identity matrix appeneded at the end of it
		*/ 
		Matrix augmentedMatrix = createIdentityAugmentedMatrix();
		augmentedMatrix.setFirstColumnIndex(1);
		augmentedMatrix.setFirstRowIndex(1);
		for (int column = 1; column <= getNumberOfColumns(); column++) {
			//  Find a row with a non-zero value in the current column, swap it into
			//  the row having index column; if we can't find a row with a non-zero value, then
			//  this matrix can't be inverted
			int targetRow = findRowWithNonZeroValueInColumn (augmentedMatrix, column, column);
			if (targetRow == augmentedMatrix.getFirstColumnIndex()-1) {
				throw new NotInvertibleException (this);
			}
		}
		//augmentedMatrix = createRREF(augmentedMatrix);
		//  Now begin the process of row reducing this matrix
		augmentedMatrix = createRREF(augmentedMatrix);
		//  The matrix where the augmented matrix was placed is the value we want
		SquareMatrix result = extractInverse (augmentedMatrix);
		result.setFirstColumnIndex(getFirstColumnIndex());
		result.setFirstRowIndex(getFirstRowIndex());
		return result;
	}


	/**
	 * returns an matrix that is the RREF representation of the given matrix.
	 * 
	 * @param matrix the matrix
	 * 
	 * @return the matrix in RREF
	 */
	public static Matrix createRREF(Matrix matrix) {
		Matrix m = new Matrix(matrix.getNumberOfRows(),matrix.getNumberOfColumns(),matrix.data);
		m.setFirstColumnIndex(matrix.getFirstColumnIndex());
		m.setFirstRowIndex(matrix.getFirstRowIndex());
		for (int row = m.getFirstRowIndex(); row <= m.getLastRowIndex(); row++) { 
			//  Find a row with a non-zero value in the current column, swap it into
			//  the row having index column; 
			int targetRow = findRowWithNonZeroValueInColumn (m, row, row);
			if(targetRow<m.getFirstRowIndex()) {
				continue;
			}
			swapRows (m, targetRow, row);
			
			//  Goal is to eliminate entries in this column for all rows below row column; 
			//  We first divide each entry in row column by the entry in this column in order to get a 
			//  1 there
			double divider = m.getValueAt(row, row);
			for (int rowColumn = row; rowColumn <= m.getNumberOfColumns(); rowColumn++) {
				m.setValueAt(row, rowColumn, m.getValueAt(row, rowColumn)/divider);
			}
			
			//  Now we need to add a multiple of this row to each row below it in order to zero-out the value in the target column
			for (int i = m.getFirstRowIndex(); i <= m.getLastRowIndex(); i++) {
				if (i != row) {
				double multiplier = -1*m.getValueAt(i, row);
					for (int rowColumn = row; rowColumn <= m.getNumberOfColumns(); rowColumn++) {
						m.setValueAt(i, rowColumn, m.getValueAt(i, rowColumn) + multiplier*m.getValueAt(row, rowColumn));
					}
				}
			}
		}
		return m;
	}

	/**
	 * Causes two rows to have their contents swapped in a matrix.
	 * 
	 * @param augmentedMatrix the matrix whose rows should be manipulated
	 * @param row1 the logical index of the first row to swap
	 * @param row2 the logical index of the second row to swap
	 */
	private static void swapRows(Matrix augmentedMatrix, int row1, int row2) {
		double[][] originalValues = augmentedMatrix.toArray();
		/*
		Matrix temp = new Matrix (augmentedMatrix.getNumberOfRows(), augmentedMatrix.getNumberOfColumns(), originalValues);
		temp.setFirstColumnIndex(augmentedMatrix.getFirstColumnIndex());
		temp.setFirstColumnIndex()
		*/
		for (int column = augmentedMatrix.getFirstColumnIndex(); column <= augmentedMatrix.getLastColumnIndex(); column++) {
			int actualColumn = augmentedMatrix.getActualColumn(column);
			augmentedMatrix.setValueAt(row2, column, originalValues[augmentedMatrix.getActualRow(row1)][actualColumn]);
			augmentedMatrix.setValueAt(row1, column, originalValues[augmentedMatrix.getActualRow(row2)][actualColumn]);
		}		
	}

	/**
	 * Finds the index of the first row with a non-zero value in the column <i>column</i>.
	 * 
	 * @param matrix the matrix to examine
	 * @param column the column for which the returned row should have a non-zero value
	 * @param startRow the row to start searching
	 * 
	 * @return the index of the first row such that matrix.getValueAt(row, column) != 0; if none are found,
	 * the return value is matrix.getFirstRowIndex()-1, which is guaranteed to not be a valid row index
	 */
	private static int findRowWithNonZeroValueInColumn(Matrix matrix, int column, int startRow) {
		for (int row = startRow; row <= matrix.getLastRowIndex(); row++) {
			if (matrix.getValueAt(row, column) != 0) {
				return row;
			}
		}
		return matrix.getFirstRowIndex()-1;
	}

	/**
	 * Given an augmented matrix, extracts the inverse from the right hand half of the matrix.
	 * 
	 * @param augmentedMatrix the augmented matrix
	 * 
	 * @return a square matrix of size augmentedMatrix.getNumberOfRows(), augmentedMatrix.getNumberOfColumns()/2
	 */
	private SquareMatrix extractInverse(Matrix augmentedMatrix) {
		SquareMatrix result = new SquareMatrix (getNumberOfRows());
		for (int row = 1; row <= augmentedMatrix.getNumberOfRows(); row++) {
			for (int column = getNumberOfColumns()+1; column <= augmentedMatrix.getLastColumnIndex(); column++) {
				result.setValueAt(row, column-getNumberOfColumns(), augmentedMatrix.getValueAt (row, column));
			}
		}
		return result;		
	}

	/**
	 * Creates the identity augmented matrix.
	 * 
	 * @return the matrix
	 */
	private Matrix createIdentityAugmentedMatrix() {
		//Matrix augmentedMatrix = new Matrix (getNumberOfRows(), getNumberOfColumns()*2);		
		Matrix identityMatrix = SquareMatrix.createIdentityMatrix(getNumberOfRows());
		//augmentedMatrix.setFirstColumnIndex(getFirstColumnIndex());
		//augmentedMatrix.setFirstRowIndex(getFirstRowIndex());
		
		return createAugmentedMatrix(this, identityMatrix);
	}

	/**
	 * Creates the augmented matrix.
	 * 
	 * @param left the left
	 * @param right the right
	 * 
	 * @return the matrix
	 */
	public static Matrix createAugmentedMatrix(Matrix left, Matrix right) {
		Matrix finalMatrix = new Matrix(left.getNumberOfRows(),left.getNumberOfColumns()+right.getNumberOfColumns());
		finalMatrix.setFirstColumnIndex(left.getFirstColumnIndex());
		finalMatrix.setFirstRowIndex(left.getFirstRowIndex());
		for (int r=left.getFirstRowIndex(); r <= left.getLastRowIndex(); r++) {
			for (int c = left.getFirstColumnIndex(); c <= left.getLastColumnIndex(); c++) {
				//  First row should give an index of 1, since that's augmentedMatrix's first row index; similarly for column
				finalMatrix.setValueAt(r-left.getFirstRowIndex()+1, c-left.getFirstColumnIndex()+1, left.getValueAt (r, c));
			}
		}
		
		for (int r = 1; r<= right.getLastRowIndex(); r++) {
			for (int c = 1; c <= right.getLastColumnIndex(); c++) {
				finalMatrix.setValueAt(r, left.getNumberOfColumns()+c, right.getValueAt(r, c));
			}
		}
		return finalMatrix;
	}
	
	public Matrix createAugmentedMatrix(Matrix right) {
		return Matrix.createAugmentedMatrix(this, right);
	}
	/**
	 * Determines whether or not this matrix is invertible.
	 * 
	 * @return <i>true</i> if the inverse of this matrix exists, <i>false</i> otherwise
	 */
	public boolean isInvertible() {
		return (getNumberOfColumns() == getNumberOfColumns()) && true;
	}
	
	/**
	 * Determines whether or not the specified index is a valid row index.
	 * 
	 * @param row the index in question
	 * 
	 * @return true if <i>row</i> is between getFirstRowIndex and getLastRowIndex;
	 * false otherwise
	 */
	private boolean isValidRow(int row)  {
		return (row >= getFirstRowIndex () && row <= getLastRowIndex ());
	}
	
	/**
	 * Determines whether or not the specified index is a valid column index.
	 * 
	 * @param column the index in question
	 * 
	 * @return true if <i>column</i> is between getFirstColumnIndex and
	 * getLastColumnIndex; false otherwise
	 */
	private boolean isValidColumn (int column) {
		return (column >= getFirstColumnIndex() && column <= getLastColumnIndex());
	}
	
	/**
	 * Throw invalid row exception.
	 * 
	 * @param row the row
	 * 
	 * @throws IndexOutOfBoundsException the index out of bounds exception
	 */
	private void throwInvalidRowException(int row) throws IndexOutOfBoundsException {
		throw new IndexOutOfBoundsException (row + " is not a valid row; " + getFirstRowIndex() + "<= row <= " + getLastRowIndex());
	}
	
	/**
	 * Throw invalid column exception.
	 * 
	 * @param col the col
	 * 
	 * @throws IndexOutOfBoundsException the index out of bounds exception
	 */
	private void throwInvalidColumnException(int col) throws IndexOutOfBoundsException {
		throw new IndexOutOfBoundsException (col+ " is not a valid column; " + getFirstColumnIndex() + "<= row <= " + getLastColumnIndex());
	}
	
	/**
	 * Computes the array row index for the specified logical index.
	 * 
	 * @param row a logical row index
	 * 
	 * @return the row index used to access the data array for the
	 * logical row <i>row</i>
	 */
	protected int getActualRow (int row) {
		return -1*getFirstRowIndex()+row;
	}
	
	/**
	 * Computes the array column index for the specified logical index.
	 * 
	 * @param column a logical column index
	 * 
	 * @return the column index used to access the data array for the
	 * logical column <i>column</i>
	 */
	protected int getActualColumn (int column) {
		return -1*getFirstColumnIndex()+column;
	}
	
	/**
	 * Sets the column.
	 * 
	 * @param column the column
	 * @param v the v
	 * 
	 * @throws IncompatibleMatrixException the incompatible matrix exception
	 */
	public void setColumn(int column, Vector v) {
		if(v.getNumberOfRows()!=this.getNumberOfColumns()) {
			throw new IncompatibleMatrixException(this,v);
		}
		for(int i = 1;i<=this.getNumberOfRows();i++) {
			this.setValueAt(i, column, v.getValueAt(i));
		}
	}
	public ColumnVector getColumn(int column) {
		ColumnVector cv = new ColumnVector(this.getNumberOfRows());
		cv.setFirstRowIndex(this.getFirstRowIndex());
		for(int i = this.getFirstRowIndex();i<=this.getLastRowIndex();i++) {
			cv.setValueAt(i, this.getValueAt(i, column));
		}
		return cv;
	}
}
